function parameters = set_parameters_aberrations
% This function sets all parameters for vectorial PSF calculations
%
% Sjoerd Stallinga, TU Delft

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands   

% parameters: NA, refractive indices of medium, cover slip, immersion fluid,
% wavelength (in nm), nominal emitter position (in nm) with z-position from
% cover slip-medium interface, spot footprint (in nm), axial range (in nm),
% sampling in pupil with (must be even), sampling in image plane (must be odd),
% sampling in axial direction, emission or excitation light path.

parameters.NA = 1.49;
parameters.refmed = 1.518;
parameters.refcov = 1.518;
parameters.refimm = 1.518;
parameters.refimmnom = parameters.refcov;
parameters.fwd = 140e3;
parameters.depth = 0;
parameters.zrange = [-800,800];
parameters.zspread = [-80,80];
% parameters.ztype = 'medium';
parameters.ztype = 'stage';
parameters.lambda = 552.0;
parameters.lambdacentral = 552;
parameters.lambdaspread = [552,552];
parameters.xemit = 0.0;
parameters.yemit = 0.0;
parameters.zemit = 0.0;
parameters.Npupil = 128;
parameters.pixelsize = 80;
parameters.Mx = 31;
parameters.My = 31;
parameters.Mz = 21;
parameters.Mlambda = 1;
parameters.Ncfg = 1;
parameters.xrange = parameters.pixelsize*parameters.Mx/2;
parameters.yrange = parameters.pixelsize*parameters.My/2;

% sanity check on position emitter w.r.t. cover slip
if strcmp(parameters.ztype,'stage')
  zcheck = parameters.depth+parameters.zemit;
end
if strcmp(parameters.ztype,'medium')
  zmin = parameters.zrange(1);
  zcheck = zmin+parameters.depth+parameters.zemit;
end
if (zcheck<-eps)
  fprintf('Warning! Emitter must be above the cover slip:\n')
  fprintf('Adjust parameter settings for physical results.\n')  
end

% sanity check on refractive index values
if (parameters.NA>parameters.refimm)
  fprintf('Warning! Refractive index immersion medium cannot be smaller than NA.\n')
end
if (parameters.NA>parameters.refcov)
  fprintf('Warning! Refractive index cover slip cannot be smaller than NA.\n')
end

% parameters needed for fixed dipole PSF only: emitter/absorber dipole
% orientation (characterized by angles pola and azim), detection/illumination
% polarization in objective lens back aperture (characterized by angles
% alpha and beta).
parameters.dipoletype = 'free';
% parameters.dipoletype = 'fixed';
parameters.pola = 45.0*pi/180;
parameters.azim = 0.0*pi/180;
parameters.polarizationpupil = false;
parameters.alpha = 45.0*pi/180;
parameters.beta = 45.0*pi/180;

% aberrations (Zernike orders [n1,m1,A1,n2,m2,A2,...] with n1,n2,... the
% radial orders, m1,m2,... the azimuthal orders, and A1,A2,... the Zernike
% coefficients in lambda rms, so 0.072 means diffraction limit)

parameters.aberrationcorrected = false;
parameters.aberrationsoffset = [];

% parameters.aberrations = [2,-2,0.0; 2,2,0.0; 3,-1,0.0; 3,1,0.0; 4,0,0.0; 3,3,0.0; 4,2,0.0; 4,4,0.0];

% up to primary, all modes 2<n+abs(m)<=4 
% parameters.aberrations = [2,-2,0.0; 2,2,0.0; 3,-1,0.0; 3,1,0.0; 4,0,0.0];
% up to secondary, all modes 2<n+abs(m)<=6
% parameters.aberrations = [2,-2,0.0; 2,2,0.0; 3,-1,0.0; 3,1,0.0; 4,0,0.0; 3,-3,-0.0; 3,3,0.0; 4,-2,0.0; 4,2,0.0; 5,-1,0.0; 5,1,0.0; 6,0,0.0];
% up to tertiary, all modes 2<n+abs(m)<=8
parameters.aberrations = [2,-2,0.0; 2,2,0.0; 3,-1,0.0; 3,1,0.0; 4,0,0.0; 3,-3,0.0; 3,3,0.0; 4,-2,0.0; 4,2,0.0; 5,-1,0.0; 5,1,0.0; 6,0,0.0; 4,-4,0.0; 4,4,0.0;  5,-3,0.0; 5,3,0.0;  6,-2,0.0; 6,2,0.0; 7,1,0.0; 7,-1,0.0; 8,0,0.0];
% up to quaternary, all modes 2<n+abs(m)<=10
% parameters.aberrations = [2,-2,0.0; 2,2,0.0; 3,-1,0.0; 3,1,0.0; 4,0,0.0; 3,-3,-0.0; 3,3,0.0; 4,-2,0.0; 4,2,0.0; 5,-1,0.0; 5,1,0.0; 6,0,0.0; 4,-4,0.0; 4,4,0.0;  5,-3,0.0; 5,3,0.0;  6,-2,0.0; 6,2,0.0; 7,1,0.0; 7,-1,0.0; 8,0,0.0; 5,-5,0.0; 5,5,0.0; 6,-4,0.0; 6,4,0.0; 7,-3,0.0; 7,3,0.0; 8,-2,0.0; 8,2,0.0; 9,-1,0.0; 9,1,0.0; 10,0,0.0];
% up to quinary, all modes 2<n+abs(m)<=12
% parameters.aberrations = [2,-2,0.0; 2,2,0.0; 3,-1,0.0; 3,1,0.0; 4,0,0.0; 3,-3,-0.0; 3,3,0.0; 4,-2,0.0; 4,2,0.0; 5,-1,0.0; 5,1,0.0; 6,0,0.0; 4,-4,0.0; 4,4,0.0;  5,-3,0.0; 5,3,0.0;  6,-2,0.0; 6,2,0.0; 7,1,0.0; 7,-1,0.0; 8,0,0.0; 5,-5,0.0; 5,5,0.0; 6,-4,0.0; 6,4,0.0; 7,-3,0.0; 7,3,0.0; 8,-2,0.0; 8,2,0.0; 9,-1,0.0; 9,1,0.0; 10,0,0.0; 6,-6,0.0; 6,6,0.0; 7,-5,0.0; 7,5,0.0; 8,-4,0.0; 8,4,0.0; 9,-3,0.0; 9,3,0.0; 10,-2,0.0; 10,2,0.0; 12,0,0.0];
% up to senary, all modes 2<n+abs(m)<=12
% parameters.aberrations = [2,-2,0.0; 2,2,0.0; 3,-1,0.0; 3,1,0.0; 4,0,0.0; 3,-3,-0.0; 3,3,0.0; 4,-2,0.0; 4,2,0.0; 5,-1,0.0; 5,1,0.0; 6,0,0.0; 4,-4,0.0; 4,4,0.0;  5,-3,0.0; 5,3,0.0;  6,-2,0.0; 6,2,0.0; 7,1,0.0; 7,-1,0.0; 8,0,0.0; 5,-5,0.0; 5,5,0.0; 6,-4,0.0; 6,4,0.0; 7,-3,0.0; 7,3,0.0; 8,-2,0.0; 8,2,0.0; 9,-1,0.0; 9,1,0.0; 10,0,0.0; 6,-6,0.0; 6,6,0.0; 7,-5,0.0; 7,5,0.0; 8,-4,0.0; 8,4,0.0; 9,-3,0.0; 9,3,0.0; 10,-2,0.0; 10,2,0.0; 12,0,0.0; 7,-5,0.0; 7,5,0.0; 8,-6,0.0; 8,6,0.0; 9,-5,0.0; 9,5,0.0; 10,-4,0.0; 10,4,0.0; 12,-2,0.0; 12,2,0.0; 14,0,0.0];


parameters.aberrations(:,3) =  parameters.aberrations(:,3)*parameters.lambdacentral;

% Diffractive Optical Element (DOE) parameters: type (binary, blaze,
% sinusiodal, interpolant), phase depth, zone function (as Zernike list)

parameters.doetype = 'none';
% parameters.doetype = 'binary';
% parameters.doetype = 'blazed';
% parameters.doetype = 'sinusoidal';
% parameters.doetype = 'spindle';
% parameters.doetype = 'azimuthramp';
% parameters.doephasedepth = 0.5*parameters.lambdacentral;
% parameters.interpolant = [0.0 0.1 0.2 0.3]*parameters.lambdacentral;
% numsingularpoints = 9;
% singularpointspacing = 0.633;
% singlength = singularpointspacing*(numsingularpoints-1)/2;
% parameters.singularpoints = linspace(-singlength,singlength,numsingularpoints);
% numzones = 2;
% parameters.ringradius = (linspace(0,1,numzones+1)).^(1/2);
% parameters.ringradius = [0.0,0.4627,0.9044,1.0];
% parameters.ringradius = [0.0,0.8887,1.0];
% parameters.zonefunction = [0,0,0.0; 1,1,0.0; 1,-1,0.7; 2,0,0.0; 4,0,-0.0; 2,-2,0; 4,-2,-0.0];

% Fit model parameters: signal photon count, background photons/pixel, read
% noise variance for sCMOS camera's, fit model, output labels depending on
% fit model

parameters.signalphotoncount = 12000;
parameters.backgroundphotoncount = 10;
parameters.readnoisestd = 0;

% fit model
parameters.fitmodel = 'aberrations';
parameters.numparams = 5+size(parameters.aberrations,1);
parameters.outputlabels = {'x','y','z','N_p_h','b_g'};
parameters.outputunits = {' nm',' nm',' nm',' ',' '};
parameters.weightvector = [1 1 1 0 0];
parameters.weightvector = parameters.weightvector/sum(parameters.weightvector);

% Bead parameters for convolution with PSF and derivatives, beaddiameter in nm
parameters.bead = true;
parameters.beaddiameter = 45;

% check on meaningfullness of bead convolution
if parameters.beaddiameter<=parameters.pixelsize
  parameters.bead = false;
end

% show intermediate results for monitoring code
parameters.debugmode = 0;

end

